﻿using System;
using System.Reflection;
using System.Threading;
using System.Threading.Tasks;
using gov.va.med.vbecs.Common.Log;

namespace VBECS.Services.Common.Timing
{
    public abstract class ServerBase : IServer, IDisposable
    {
        readonly CancellationTokenSource _tokenSource = new CancellationTokenSource();
        private bool _disposed; // to detect redundant calls
        Task _task;
        public abstract event EventHandler<EventArgs> TimeoutElapsed;
        public event EventHandler<ThreadExceptionEventArgs> FatalErrorOccured;
        // Logger
        protected readonly ILogger Logger =
            LogManager.Instance().LoggerLocator.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);

        protected ServerBase(int timeOut)
        {
            TimeOut = timeOut;
        }

        public void Start()
        {
            // Run worker process in the separate thread
            // Task.Run(() => do_start(_tokenSource.Token), _tokenSource.Token);
            _task = Task.Factory.StartNew(() => do_start(_tokenSource.Token), _tokenSource.Token).ContinueWith(
            t =>
                {
                    // Report to caller thread
                    if (null != FatalErrorOccured) 
                        FatalErrorOccured(this, t.Exception == null ? null : new ThreadExceptionEventArgs(t.Exception.InnerException)); 
                    // There is no need to dispose task it was already completed
                    _tokenSource.Cancel();
                    _task = null;
                }, TaskContinuationOptions.OnlyOnFaulted);
        }

        public void Stop()
        {
            Dispose(true);
        }

        protected abstract void do_start(CancellationToken token);

        public int TimeOut { get; private set; }

        #region Disposing

        // Perform any object clean up here.
        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        virtual protected void Dispose(bool disposing)
        {
            if (!_disposed)
            {
                if (disposing)
                {
                    _tokenSource.Cancel();
                    //wait before task stops
                    if (_task != null)
                    {
                        try { _task.Wait(); }
                        catch (AggregateException ex)
                        {
                            // Any other exception besides OperationCanceledException will be re-throwed in new AggregateException
                            // http://www.blackwasp.co.uk/WithoutCancellations.aspx
                            ex.Handle(e => e is OperationCanceledException);
                        }
                        _task.Dispose();
                        _task = null;
                    }
                }

                // shared cleanup logic
                _disposed = true;
            }
        }

        ~ServerBase()
        {
            Dispose(false);
        }
        
        #endregion
    }
}
